/*
This file is part of mfaktc.
Copyright (C) 2009, 2010, 2011, 2012  Oliver Weihe (o.weihe@t-online.de)
                                      Bertram Franz (bertramf@gmx.net)

mfaktc is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

mfaktc is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                
You should have received a copy of the GNU General Public License
along with mfaktc.  If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <cuda_runtime.h>
#include <time.h>

#include "params.h"
#include "my_types.h"
#include "output.h"
#include "compatibility.h"
#include "tf_validate.h"

unsigned long long int calculate_k_min(int fermat, unsigned int exp, int bits);	// From mfaktc.c
unsigned long long int calculate_k_max(int fermat, unsigned int exp, int bits);

void print_help(char *string)
{
  printf("mmff v%s Copyright (C) 2009, 2010, 2011, 2012  George Woltman, Oliver Weihe\n", MFAKTC_VERSION);
  printf("This program comes with ABSOLUTELY NO WARRANTY; for details see COPYING.\n");
  printf("This is free software, and you are welcome to redistribute it\n");
  printf("under certain conditions; see COPYING for details.\n\n\n");

  printf("Usage: %s [options]\n", string);
  printf("  -h                     display this help and exit\n");
  printf("  -d <device number>     specify the device number used by this program\n");
  printf("  -v <number>            set verbosity (min = 0, default = 1, max = 3)\n");
  printf("\n");
}


/*
print_dezXXX(intXXX a, char *buf) writes "a" into "buf" in decimal
"buf" must be preallocated with enough space.
Enough space is
  30 bytes for print_dez96()   (2^96 -1  has 29 decimal digits)
  59 bytes for print_dez192()  (2^192 -1 has 58 decimal digits)
  156 bytes for print_dez512() (2^512 -1 has 155 decimal digits)
*/

void print_dez96(int96 a, char *buf)
{
  int192 tmp;

  tmp.d5 = 0;
  tmp.d4 = 0;
  tmp.d3 = 0;
  tmp.d2 = a.d2;
  tmp.d1 = a.d1;
  tmp.d0 = a.d0;

  print_dez192(tmp, buf);
}


void print_dez192(int192 a, char *buf)
{
  char digit[58];
  int digits=0,carry,i=0;
  long long int tmp;
  
  while((a.d0!=0 || a.d1!=0 || a.d2!=0 || a.d3!=0 || a.d4!=0 || a.d5!=0) && digits<58)
  {
                                                   carry = a.d5%10; a.d5 /= 10;
    tmp = a.d4; tmp += (long long int)carry << 32; carry = tmp%10;  a.d4 =  tmp/10;
    tmp = a.d3; tmp += (long long int)carry << 32; carry = tmp%10;  a.d3 =  tmp/10;
    tmp = a.d2; tmp += (long long int)carry << 32; carry = tmp%10;  a.d2 =  tmp/10;
    tmp = a.d1; tmp += (long long int)carry << 32; carry = tmp%10;  a.d1 =  tmp/10;
    tmp = a.d0; tmp += (long long int)carry << 32; carry = tmp%10;  a.d0 =  tmp/10;
    digit[digits++] = carry;
  }
  if(digits == 0)sprintf(buf, "0");
  else
  {
    digits--;
    while(digits >= 0)
    {
      sprintf(&(buf[i++]), "%1d", digit[digits--]);
    }
  }
}


void print_dez512(int512 a, char *buf)
{
  char digit[155];
  int digits=0,carry,i=0;
  long long int tmp;
  
  while(((a.d0 | a.d1 | a.d2 | a.d3 | a.d4 | a.d5 | a.d6 | a.d7 | a.d8 | a.d9 | a.d10 | a.d11 | a.d12 | a.d13 | a.d14 | a.d15) != 0) && digits<155)
  {
                                                    carry = a.d15%10; a.d15 /= 10;
    tmp = a.d14; tmp += (long long int)carry << 32; carry = tmp%10;  a.d14 =  tmp/10;
    tmp = a.d13; tmp += (long long int)carry << 32; carry = tmp%10;  a.d13 =  tmp/10;
    tmp = a.d12; tmp += (long long int)carry << 32; carry = tmp%10;  a.d12 =  tmp/10;
    tmp = a.d11; tmp += (long long int)carry << 32; carry = tmp%10;  a.d11 =  tmp/10;
    tmp = a.d10; tmp += (long long int)carry << 32; carry = tmp%10;  a.d10 =  tmp/10;
    tmp = a.d9; tmp += (long long int)carry << 32; carry = tmp%10;  a.d9 =  tmp/10;
    tmp = a.d8; tmp += (long long int)carry << 32; carry = tmp%10;  a.d8 =  tmp/10;
    tmp = a.d7; tmp += (long long int)carry << 32; carry = tmp%10;  a.d7 =  tmp/10;
    tmp = a.d6; tmp += (long long int)carry << 32; carry = tmp%10;  a.d6 =  tmp/10;
    tmp = a.d5; tmp += (long long int)carry << 32; carry = tmp%10;  a.d5 =  tmp/10;
    tmp = a.d4; tmp += (long long int)carry << 32; carry = tmp%10;  a.d4 =  tmp/10;
    tmp = a.d3; tmp += (long long int)carry << 32; carry = tmp%10;  a.d3 =  tmp/10;
    tmp = a.d2; tmp += (long long int)carry << 32; carry = tmp%10;  a.d2 =  tmp/10;
    tmp = a.d1; tmp += (long long int)carry << 32; carry = tmp%10;  a.d1 =  tmp/10;
    tmp = a.d0; tmp += (long long int)carry << 32; carry = tmp%10;  a.d0 =  tmp/10;
    digit[digits++] = carry;
  }
  if(digits == 0)sprintf(buf, "0");
  else
  {
    digits--;
    while(digits >= 0)
    {
      sprintf(&(buf[i++]), "%1d", digit[digits--]);
    }
  }
}


void print_timestamp(FILE *outfile)
{
  time_t now;
  char *ptr;

  now = time(NULL);
  ptr = ctime(&now);
  ptr[24] = '\0'; // cut off the newline
  fprintf(outfile, "[%s]\n", ptr);
}


void print_status_line(mystuff_t *mystuff)
{
  unsigned long long int eta;
  int i = 0, max_class_number;
  char buffer[256];
  int index = 0;
  time_t now;
  struct tm *tm_now = NULL;
  int time_read = 0;
                        

  if(mystuff->mode == MODE_SELFTEST_SHORT) return; /* no output during short selftest */
  
#ifdef MORE_CLASSES
  max_class_number = 960;
#else
  max_class_number = 96;
#endif

  if(mystuff->stats.output_counter == 0)
  {
    if (mystuff->stats.cpu_wait == -2.0f)		// Hack to indicate GPU sieving kernel
      printf("%s\n", mystuff->stats.gpuprogressheader);
//    else						// CPU sieving
//      printf("%s\n", mystuff->stats.progressheader);
    mystuff->stats.output_counter = 20;
  }
  if(mystuff->printmode == 0)mystuff->stats.output_counter--;
  
  while(mystuff->stats.progressformat[i] && i < 250)
  {
    if(mystuff->stats.progressformat[i] != '%')
    {
      buffer[index++] = mystuff->stats.progressformat[i];
      i++;
    }
    else
    {
      if(mystuff->stats.progressformat[i+1] == 'C')
      {
        index += sprintf(buffer + index, "%4d", mystuff->stats.class_number);
      }
      else if(mystuff->stats.progressformat[i+1] == 'c')
      {
        index += sprintf(buffer + index, "%3d", mystuff->stats.class_counter);
      }
      else if(mystuff->stats.progressformat[i+1] == 'p')
      {
        index += sprintf(buffer + index, "%5.1f", (double)(mystuff->stats.class_counter * 100) / (double)max_class_number);
      }
      else if(mystuff->stats.progressformat[i+1] == 'g')
      {
        if (mystuff->stats.cpu_wait == -2.0f)		// Hack to indicate GPU sieving kernel
          index += sprintf(buffer + index, "  n.a. ");
//	else	      
//        index += sprintf(buffer + index, "%7.2f", mystuff->stats.ghzdays * 86400000.0f / ((double)mystuff->stats.class_time * (double)max_class_number));
      }
      else if(mystuff->stats.progressformat[i+1] == 't')
      {
             if(mystuff->stats.class_time < 100000ULL  )index += sprintf(buffer + index, "%6.3f", (double)mystuff->stats.class_time/1000.0);
        else if(mystuff->stats.class_time < 1000000ULL )index += sprintf(buffer + index, "%6.2f", (double)mystuff->stats.class_time/1000.0);
        else if(mystuff->stats.class_time < 10000000ULL)index += sprintf(buffer + index, "%6.1f", (double)mystuff->stats.class_time/1000.0);
        else                                            index += sprintf(buffer + index, "%6.0f", (double)mystuff->stats.class_time/1000.0);
      }
      else if(mystuff->stats.progressformat[i+1] == 'e')
      {
        if(mystuff->mode == MODE_NORMAL)
        {
          if(mystuff->stats.class_time > 250)
          {
            eta = (mystuff->stats.class_time * (max_class_number - mystuff->stats.class_counter) + 500)  / 1000;
                 if(eta < 3600) index += sprintf(buffer + index, "%2" PRIu64 "m%02" PRIu64 "s", eta / 60, eta % 60);
            else if(eta < 86400)index += sprintf(buffer + index, "%2" PRIu64 "h%02" PRIu64 "m", eta / 3600, (eta / 60) % 60);
            else                index += sprintf(buffer + index, "%2" PRIu64 "d%02" PRIu64 "h", eta / 86400, (eta / 3600) % 24);
          }
          else                  index += sprintf(buffer + index, "  n.a.");
        }
        else if(mystuff->mode == MODE_SELFTEST_FULL)index += sprintf(buffer + index, "  n.a.");
      }
      else if(mystuff->stats.progressformat[i+1] == 'n')
      {
        if (mystuff->stats.cpu_wait == -2.0f) {		// Hack to indicate GPU sieving kernel
	  if(mystuff->stats.grid_count < (1000000000 / mystuff->gpu_sieve_processing_size + 1))
	    index += sprintf(buffer + index, "%6.2fM", (double)mystuff->stats.grid_count * mystuff->gpu_sieve_processing_size / 1000000.0);
	  else
	    index += sprintf(buffer + index, "%6.2fG", (double)mystuff->stats.grid_count * mystuff->gpu_sieve_processing_size / 1000000000.0);
	}
//	else {					// CPU sieving
//	  if(((unsigned long long int)mystuff->threads_per_grid * (unsigned long long int)mystuff->stats.grid_count) < 1000000000ULL)
//            index += sprintf(buffer + index, "%6.2fM", (double)mystuff->threads_per_grid * (double)mystuff->stats.grid_count / 1000000.0);
//          else
//		  index += sprintf(buffer + index, "%6.2fG", (double)mystuff->threads_per_grid * (double)mystuff->stats.grid_count / 1000000000.0);
//	}
      }
      else if(mystuff->stats.progressformat[i+1] == 'r')
      {
        if (mystuff->stats.cpu_wait == -2.0f)		// Hack to indicate GPU sieving kernel
	  index += sprintf(buffer + index, "%6.2f", (double)mystuff->stats.grid_count * mystuff->gpu_sieve_processing_size / ((double)mystuff->stats.class_time * 1000.0));
//	else						// CPU sieving
//          index += sprintf(buffer + index, "%6.2f", (double)mystuff->threads_per_grid * (double)mystuff->stats.grid_count / ((double)mystuff->stats.class_time * 1000.0));
      }
      else if(mystuff->stats.progressformat[i+1] == 's')
      {
        if (mystuff->stats.cpu_wait == -2.0f)		// Hack to indicate GPU sieving kernel
	  index += sprintf(buffer + index, "%7d", mystuff->gpu_sieve_primes-1);  // Output number of odd primes sieved
//	else						// CPU sieving
//          index += sprintf(buffer + index, "%7d", mystuff->sieve_primes);
      }
      else if(mystuff->stats.progressformat[i+1] == 'w')
      {
        index += sprintf(buffer + index, "(n.a.)"); /* mfakto only */
      }
      else if(mystuff->stats.progressformat[i+1] == 'W')
      {
        if(mystuff->stats.cpu_wait >= 0.0f)index += sprintf(buffer + index, "%6.2f", mystuff->stats.cpu_wait);
        else                               index += sprintf(buffer + index, " n.a.");
      }
      else if(mystuff->stats.progressformat[i+1] == 'd')
      {
        if(!time_read)
        {
          now = time(NULL);
          tm_now = localtime(&now);
          time_read = 1;
        }
        index += strftime(buffer + index, 7, "%b %d", tm_now);
      }
      else if(mystuff->stats.progressformat[i+1] == 'T')
      {
        if(!time_read)
        {
          now = time(NULL);
          tm_now = localtime(&now);
          time_read = 1;
        }
        index += strftime(buffer + index, 6, "%H:%M", tm_now);
      }
      else if(mystuff->stats.progressformat[i+1] == 'U')
      {
        index += sprintf(buffer + index, "%s", mystuff->V5UserID);
      }
      else if(mystuff->stats.progressformat[i+1] == 'H')
      {
        index += sprintf(buffer + index, "%s", mystuff->ComputerID);
      }
      else if(mystuff->stats.progressformat[i+1] == 'M')
      {
        index += sprintf(buffer + index, "%-10u", mystuff->exponent);
      }
      else if(mystuff->stats.progressformat[i+1] == 'l')
      {
        index += sprintf(buffer + index, "%2d", mystuff->bit_min);
      }
      else if(mystuff->stats.progressformat[i+1] == 'u')
      {
        index += sprintf(buffer + index, "%2d", mystuff->bit_max_stage);
      }
      else if(mystuff->stats.progressformat[i+1] == '%')
      {
        buffer[index++] = '%';
      }
      else /* '%' + unknown format character -> just print "%<character>" */
      {
        buffer[index++] = '%';
        buffer[index++] = mystuff->stats.progressformat[i];
      }

      i += 2;
    }
    if(index > 200) /* buffer has 256 bytes, single format strings are limited to 50 bytes */
    {
      buffer[index] = 0;
      printf("%s", buffer);
      index = 0;
    }
  }
  
  
  if(mystuff->mode == MODE_NORMAL)
  {
    if(mystuff->printmode == 1)index += sprintf(buffer + index, "\r");
    else                       index += sprintf(buffer + index, "\n");
  }
  if(mystuff->mode == MODE_SELFTEST_FULL && mystuff->printmode == 0)
  {
    index += sprintf(buffer + index, "\n");
  }

  buffer[index] = 0;
  printf("%s", buffer);
}


void print_result_line(mystuff_t *mystuff, int factorsfound)
/* printf the final result line (STDOUT and resultfile) */
{
  char UID[110]; /* 50 (V5UserID) + 50 (ComputerID) + 8 + spare */
  char string[250];
  unsigned long long k_min, k_max;
  char krange[100];

  FILE *resultfile=NULL;

  if(mystuff->V5UserID[0] && mystuff->ComputerID[0])
    sprintf(UID, "UID: %s/%s, ", mystuff->V5UserID, mystuff->ComputerID);
  else
    UID[0]=0;

  k_min=calculate_k_min(mystuff->fermat_factoring, mystuff->exponent, mystuff->bit_min);
  k_max=calculate_k_max(mystuff->fermat_factoring, mystuff->exponent, mystuff->bit_max_stage);
  if (mystuff->k_lower_bound > k_min) k_min = mystuff->k_lower_bound;
  if (mystuff->k_upper_bound && mystuff->k_upper_bound < k_max) k_max = mystuff->k_upper_bound;
  if (k_min % 1000000000000000ULL == 0 && k_max % 1000000000000000ULL == 0)
    sprintf(krange, "k range: %" PRIu64 "P to %" PRIu64 "P", k_min / 1000000000000000ULL, k_max / 1000000000000000ULL);
  else if (k_min % 1000000000000ULL == 0 && k_max % 1000000000000ULL == 0)
    sprintf(krange, "k range: %" PRIu64 "T to %" PRIu64 "T", k_min / 1000000000000ULL, k_max / 1000000000000ULL);
  else if (k_min % 1000000000ULL == 0 && k_max % 1000000000ULL == 0)
    sprintf(krange, "k range: %" PRIu64 "G to %" PRIu64 "G", k_min / 1000000000ULL, k_max / 1000000000ULL);
  else if (k_min % 1000000ULL == 0 && k_max % 1000000ULL == 0)
    sprintf(krange, "k range: %" PRIu64 "M to %" PRIu64 "M", k_min / 1000000ULL, k_max / 1000000ULL);
  else
    sprintf(krange, "k range: %" PRIu64 " to %" PRIu64 "", k_min, k_max);
  sprintf(krange+strlen(krange), " (%d-bit factors)", mystuff->bit_max_stage);

  if(mystuff->mode == MODE_NORMAL)
  {
    resultfile = fopen(mystuff->resultfile, "a");
    if(mystuff->print_timestamp == 1)print_timestamp(resultfile);
  }
  if(factorsfound)
  {
#ifndef MORE_CLASSES
    if((mystuff->mode == MODE_NORMAL) && (mystuff->stats.class_counter < 96))
#else
    if((mystuff->mode == MODE_NORMAL) && (mystuff->stats.class_counter < 960))
#endif
    {
      sprintf(string, "found %d factor%s for %s in %s (partially tested) [mmff %s %s]", factorsfound, (factorsfound > 1) ? "s" : "", mystuff->exponent_string, krange, MFAKTC_VERSION, mystuff->stats.kernelname);
    }
    else
    {
      sprintf(string, "found %d factor%s for %s in %s [mmff %s %s]", factorsfound, (factorsfound > 1) ? "s" : "", mystuff->exponent_string, krange, MFAKTC_VERSION, mystuff->stats.kernelname);
    }
  }
  else
  {
    sprintf(string, "no factor for %s in %s [mmff %s %s]", mystuff->exponent_string, krange, MFAKTC_VERSION, mystuff->stats.kernelname);
  }

  if(mystuff->mode != MODE_SELFTEST_SHORT)
  {
    printf("%s\n", string);
  }
  if(mystuff->mode == MODE_NORMAL)
  {
    fprintf(resultfile, "%s%s\n", UID, string);
    fclose(resultfile);
  }
}


void print_factor(mystuff_t *mystuff, int factor_number, char *factor)
{
  char UID[110]; /* 50 (V5UserID) + 50 (ComputerID) + 8 + spare */
  FILE *resultfile = NULL;
  char exponent_string[80];

  if(mystuff->V5UserID[0] && mystuff->ComputerID[0])
    sprintf(UID, "UID: %s/%s, ", mystuff->V5UserID, mystuff->ComputerID);
  else
    UID[0]=0;

  if (mystuff->fermat_factoring)	// Figure out which Fermat number this factor divides
  {
    sprintf(exponent_string, "F%d", which_fermat_number(mystuff, factor_number));
  }
  else
    sprintf(exponent_string, "%s", mystuff->exponent_string);

  if(mystuff->mode == MODE_NORMAL)
  {
    resultfile = fopen(mystuff->resultfile, "a");
    if(mystuff->print_timestamp == 1 && factor_number == 0)print_timestamp(resultfile);
  }

  if(factor_number < 10)
  {
    char k[155]; int carry, i, l, N;

    /* SB: don't want to mess with lower functions; I will simply do the k calculation here on a decimal string */
    /* factors are extremely rare, anyway */

    l = strlen(factor)-1;
    memcpy(k, factor,l+2);
    for(N=0; N==0 || (k[l]%2)==0; N++) {      /* factor = "k*2^N+1"; disregard last odd digit once */
      for(i=carry=0; i<=l; i++) {
        int d = k[i] - '0' + 10 * carry;
        carry = d & 1;
        k[i]  = d / 2 + '0';
      }
    }
    l++; /* now it is strlen */
    for(i=0; k[i]=='0'; i++);         /* squeeze leading zeroes */
    if(i) { l -= i; memmove(k, k+i, l); }
    sprintf(k+l, "*2^%d+1", N);

    if(mystuff->mode != MODE_SELFTEST_SHORT)
    {
      if(mystuff->printmode == 1 && factor_number == 0)printf("\n");
      printf("%s has a factor: %s\n", exponent_string, k);
    }
    if(mystuff->mode == MODE_NORMAL)
    {
#ifndef MORE_CLASSES
      fprintf(resultfile, "%s%s has a factor: %s = %s [TF:%d:%d%s:mmff %s %s]\n", UID, exponent_string, k, factor, mystuff->bit_min, mystuff->bit_max_stage, ((mystuff->stopafterfactor == 2) && (mystuff->stats.class_counter <  96)) ? "*" : "" , MFAKTC_VERSION, mystuff->stats.kernelname);
#else
      fprintf(resultfile, "%s%s has a factor: %s = %s [TF:%d:%d%s:mmff %s %s]\n", UID, exponent_string, k, factor, mystuff->bit_min, mystuff->bit_max_stage, ((mystuff->stopafterfactor == 2) && (mystuff->stats.class_counter < 960)) ? "*" : "" , MFAKTC_VERSION, mystuff->stats.kernelname);
#endif
    }
  }
  else /* factor_number >= 10 */
  {
    if(mystuff->mode != MODE_SELFTEST_SHORT)      printf("%s: %d additional factors not shown\n",      mystuff->exponent_string, factor_number-10);
    if(mystuff->mode == MODE_NORMAL)fprintf(resultfile,"%s%s: %d additional factors not shown\n", UID, mystuff->exponent_string, factor_number-10);
  }

  if(mystuff->mode == MODE_NORMAL)fclose(resultfile);
}

