/*
This file is part of mfaktc.
Copyright (C) 2009, 2010, 2011, 2012  Oliver Weihe (o.weihe@t-online.de)

mfaktc is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

mfaktc is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with mfaktc.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "gpusieve.h"
#include "timer.h"

extern "C" __host__ int tf_class_barrett92_gs(unsigned long long int k_min, unsigned long long int k_max, mystuff_t *mystuff)
{
  int i;
  timeval timer;
  int96 k_base;
  int count = 0;
  int numblocks;
  unsigned long long true_k_remaining, rounded_k_remaining;
  int shared_mem_required;
  int factorsfound = 0;
  int validation_counter = (mystuff->verbosity >= 3) ? 10 : 2;

  // If we've never initialized the GPU sieving code, do so now.
  gpusieve_init (mystuff);

  // If we haven't initialized the GPU sieving code for this Mersenne exponent, do so now.
  gpusieve_init_exponent (mystuff);

  // Init the timer
  timer_init(&timer);

  // set result array to 0
  cudaMemset(mystuff->d_RES, 0, 1*sizeof(int)); //first int of result array contains the number of factors found

  // Calculate the initial bit-to-clear values for this class
  gpusieve_init_class (mystuff, k_min);

  // Generously estimate the shared memory requirements for the TF kernel
#ifdef RAW_GPU_BENCH
  shared_mem_required = 100;						// no sieving = 100%
#else
  if (mystuff->gpu_sieve_primes < 54) shared_mem_required = 100;	// no sieving = 100%
  else if (mystuff->gpu_sieve_primes < 310) shared_mem_required = 50;	// 54 primes expect 48.30%
  else if (mystuff->gpu_sieve_primes < 1846) shared_mem_required = 38;	// 310 primes expect 35.50%
  else if (mystuff->gpu_sieve_primes < 21814) shared_mem_required = 30;	// 1846 primes expect 28.10%
  else if (mystuff->gpu_sieve_primes < 67894) shared_mem_required = 24;	// 21814 primes expect 21.93%
  else shared_mem_required = 22;					// 67894 primes expect 19.94%
#endif
  shared_mem_required = mystuff->gpu_sieve_processing_size * sizeof (int) * shared_mem_required / 100;

  // Loop until all the k's are processed
  for(;;)
  {

    // Calculate the number of k's remaining.  Round this up so that we sieve an array that is
    // a multiple of the bits processed by each TF kernel (my_stuff->gpu_sieve_processing_size).

    true_k_remaining = ((k_max - k_min + 1) + NUM_CLASSES - 1) / NUM_CLASSES;
    if (true_k_remaining < (unsigned long long) mystuff->gpu_sieve_size) {
      if (true_k_remaining == 0) break; // Can happen on tiny k ranges like "FermatFactor=62,64,65"
      numblocks = ((int) true_k_remaining + mystuff->gpu_sieve_processing_size - 1) / mystuff->gpu_sieve_processing_size;
      rounded_k_remaining = numblocks * mystuff->gpu_sieve_processing_size;
    } else {
      numblocks = mystuff->gpu_sieve_size / mystuff->gpu_sieve_processing_size;
      rounded_k_remaining = true_k_remaining;
    }

    // Do some sieving on the GPU.

    gpusieve (mystuff, rounded_k_remaining);

    // Clear excess bits in the sieve (they can cause bogus exponentiation failure errors)

    if (true_k_remaining != rounded_k_remaining) gpusieve_clear_excess (mystuff, true_k_remaining, rounded_k_remaining);

    // Set the k value corresponding to the first bit in the bit array

    k_base.d0 = (int) (k_min & 0xFFFFFFFF);
    k_base.d1 = (int) (k_min >> 32);
    k_base.d2 = 0;

    // Now let the GPU trial factor the candidates that survived the sieving

    if (mystuff->fermat_factoring) {
      if (mystuff->exponent <= 31) {
        if (mystuff->bit_max_stage <= 89)
	  mfaktc_barrett89_F0_31gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-63);
        else if (mystuff->bit_max_stage <= 96)
	  mfaktc_barrett96_F0_31gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
	else
          printf ("1 No kernel for that bit level\n"), exit (1);
      }
      else if (mystuff->exponent <= 63) {
        if (mystuff->bit_max_stage <= 89)
	  mfaktc_barrett89_F32_63gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-63);
        else if (mystuff->bit_max_stage <= 96)
	  mfaktc_barrett96_F32_63gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 108)
	  mfaktc_barrett108_F32_63gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 120)
	  mfaktc_barrett120_F32_63gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+32));
        else if (mystuff->bit_max_stage <= 128)
	  mfaktc_barrett128_F32_63gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
	else
          printf ("2 No kernel for that bit level\n"), exit (1);
      }
      else if (mystuff->exponent <= 95) {
        if (mystuff->bit_max_stage <= 108)
	  mfaktc_barrett108_F64_95gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 120)
	  mfaktc_barrett120_F64_95gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+32));
        else if (mystuff->bit_max_stage <= 128)
	  mfaktc_barrett128_F64_95gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 140)
	  mfaktc_barrett140_F64_95gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 152)
	  mfaktc_barrett152_F64_95gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+64));
        else if (mystuff->bit_max_stage <= 160)
	  mfaktc_barrett160_F64_95gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
	else
          printf ("3 No kernel for that bit level\n"), exit (1);
      }
      else if (mystuff->exponent <= 127) {
        if (mystuff->bit_max_stage <= 140)
	  mfaktc_barrett140_F96_127gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 152)
	  mfaktc_barrett152_F96_127gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+64));
        else if (mystuff->bit_max_stage <= 160)
	  mfaktc_barrett160_F96_127gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 172)
	  mfaktc_barrett172_F96_127gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 183)
	  mfaktc_barrett183_F96_127gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+96));
	else
          printf ("4 No kernel for that bit level\n"), exit (1);
      }
      else if (mystuff->exponent <= 159) {
        if (mystuff->bit_max_stage <= 172)
	  mfaktc_barrett172_F128_159gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 183)
	  mfaktc_barrett183_F128_159gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+96));
        else if (mystuff->bit_max_stage <= 185)
	  mfaktc_barrett185_F128_159gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+96));
        else if (mystuff->bit_max_stage <= 188)
	  mfaktc_barrett188_F128_159gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+96));
        else if (mystuff->bit_max_stage <= 192)
	  mfaktc_barrett192_F128_159gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 204)
	  mfaktc_barrett204_F128_159gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 215)
	  mfaktc_barrett215_F128_159gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+128));
	else
          printf ("5 No kernel for that bit level\n"), exit (1);
      }
      else if (mystuff->exponent <= 191) {
        if (mystuff->bit_max_stage <= 204)
	  mfaktc_barrett204_F160_191gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 215)
	  mfaktc_barrett215_F160_191gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+128));
        else if (mystuff->bit_max_stage <= 217)
	  mfaktc_barrett217_F160_191gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+128));
        else if (mystuff->bit_max_stage <= 220)
	  mfaktc_barrett220_F160_191gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+128));
        else if (mystuff->bit_max_stage <= 224)
	  mfaktc_barrett224_F160_191gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 236)
	  mfaktc_barrett236_F160_191gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 247)
	  mfaktc_barrett247_F160_191gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+160));
	else
          printf ("6 No kernel for that bit level (%d %d)\n", mystuff->exponent,mystuff->bit_max_stage), exit (1);
      }
      else if (mystuff->exponent <= 223) {
        if (mystuff->bit_max_stage <= 236)
	  mfaktc_barrett236_F192_223gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
        else if (mystuff->bit_max_stage <= 247)
	  mfaktc_barrett247_F192_223gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+160));
        else if (mystuff->bit_max_stage <= 249)
	  mfaktc_barrett249_F192_223gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+160));
        else if (mystuff->bit_max_stage <= 252)
	  mfaktc_barrett252_F192_223gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(mystuff->exponent, k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+160));
	else
          printf ("No kernel for that bit level (%d %d)\n", mystuff->exponent,mystuff->bit_max_stage), exit (1);
      }
      else
        printf ("No kernel for that Fermat number\n"), exit (1);
    }
    else if (mystuff->exponent == 31) {
      if (mystuff->bit_max_stage <= 89)
	mfaktc_barrett89_M31gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-63);
      else if (mystuff->bit_max_stage <= 96)
        mfaktc_barrett96_M31gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
      else
	printf ("No kernel for that bit level\n"), exit (1);
    } else if (mystuff->exponent == 61) {
      if (mystuff->bit_max_stage <= 108)
	mfaktc_barrett108_M61gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
      else if (mystuff->bit_max_stage <= 120)
        mfaktc_barrett120_M61gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+32));
      else if (mystuff->bit_max_stage <= 125)
        mfaktc_barrett128_M61gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
      else
	printf ("No kernel for that bit level\n"), exit (1);
    } else if (mystuff->exponent == 89) {
      if (mystuff->bit_max_stage <= 128)
	mfaktc_barrett128_M89gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
      else if (mystuff->bit_max_stage <= 140)
        mfaktc_barrett140_M89gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
      else if (mystuff->bit_max_stage <= 152)
        mfaktc_barrett152_M89gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+64));
      else if (mystuff->bit_max_stage <= 153)
        mfaktc_barrett160_M89gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
      else
	printf ("No kernel for that bit level\n"), exit (1);
    } else    if (mystuff->exponent == 107) {
      if (mystuff->bit_max_stage <= 152)
	mfaktc_barrett152_M107gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+64));
      else if (mystuff->bit_max_stage <= 160)
        mfaktc_barrett160_M107gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
      else if (mystuff->bit_max_stage <= 172)
        mfaktc_barrett172_M107gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES);
      else
	printf ("No kernel for that bit level\n"), exit (1);
    } else if (mystuff->exponent == 127) {
      if (mystuff->bit_max_stage <= 183)
	mfaktc_barrett183_M127gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+96));
      else if (mystuff->bit_max_stage <= 185)
        mfaktc_barrett185_M127gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+96));
      else if (mystuff->bit_max_stage <= 188)
        mfaktc_barrett188_M127gs<<<numblocks, THREADS_PER_BLOCK, shared_mem_required>>>(k_base, mystuff->d_bitarray, mystuff->gpu_sieve_processing_size, mystuff->d_RES, mystuff->bit_min-(63+96));
      else
	printf ("No kernel for that bit level\n"), exit (1);
    }
    else
        printf ("No such kernel!\n"), exit (1);

    // Sync before doing more GPU sieving
    cudaThreadSynchronize();

    // Count the number of blocks processed
    count += numblocks;

    // Now do verification of one of the modular exponentiations

    if (validation_counter) {
      validation_counter--;
      validate_exponentiation(mystuff, true_k_remaining);
      if (validation_counter == 0 && mystuff->verbosity == 88) break;
    }

    // Move to next batch of k's
    k_min += (unsigned long long) mystuff->gpu_sieve_size * NUM_CLASSES;
    if (k_min > k_max) break;

    // Call a different routine to advance the bit-to-clear values by gpusieve_size bits
    gpusieve_continue_class (mystuff);

#ifdef TESTIT
static int compare1[100000];
static int compare2[100000];
cudaMemcpy (compare1, mystuff->d_sieve_info, 100000*sizeof(int), cudaMemcpyDeviceToHost);
gpusieve_init_class (mystuff, k_min);
cudaMemcpy (compare2, mystuff->d_sieve_info, 100000*sizeof(int), cudaMemcpyDeviceToHost);
    if (memcmp (compare1, compare2, 100000*sizeof (int)))
	    printf ("Advance bit-to-clr failed!!\n");
//    else
//	    printf ("Advance bit-to-clr worked!!\n");
#endif
  }

/* download any factor found results from GPU */
  cudaMemcpy(mystuff->h_RES, mystuff->d_RES, RESULTS_ARRAY_VALIDATION_OFFSET*sizeof(int), cudaMemcpyDeviceToHost);

  // Set grid count to the number of blocks processed.  The print code will convert this to a
  // count of candidates processed (by multiplying by 8192 * THREADS_PER_BLOCK.
  // This count isn't an exact match to CPU sieving case as that counts candidates after sieving
  // and we are counting candidates before sieving.  We'd have to modify the TF kernels to count
  // the candidates processed to be completely compatible.
  mystuff->stats.grid_count = count;

  // Keep track of time spent TFing this class
  /* prevent division by zero if timer resolution is too low */
  mystuff->stats.class_time = timer_diff(&timer)/1000;
  if(mystuff->stats.class_time == 0)mystuff->stats.class_time = 1;

  // GPU sieving does not wait on the CPU (also used by print_status_line to indicate this is a GPU sieving kernel)
  mystuff->stats.cpu_wait = -2.0f;

  // Print out a useful status line
  print_status_line(mystuff);

  // Print out any found factors
  factorsfound=mystuff->h_RES[0];
  for(i=0; (i<factorsfound) && (i<10); i++)
  {
    unsigned int fac[32], *hptr, *facptr;
    unsigned int j, datalen;
    char string[180];

    // get the data

    datalen = mystuff->h_RES[1];

    facptr = fac;
    *facptr++ = datalen;
    hptr = mystuff->h_RES + i * datalen + 2;
    for (j = 0; j < datalen; j++) *facptr++ = *hptr++;

    dn_print(fac,string);
    print_factor(mystuff, i, string);
  }
  if(factorsfound>=10)
  {
    print_factor(mystuff, factorsfound, NULL);
  }

  return factorsfound;
}
